C      *****************************************************************
C      * Subroutine Jake                                               *
C      * The partial derivatives, for each component, of the total     *
C      * component concentration with respect to each free component   *
C      * concentration are stored in the Jacobian matrix [A]. The total*
C      * component concentration is the concentration of the component *
C      * as free component and as part of a dependent species. The eqn *
C      * [A]X=R is formed where [A] is the matrix of the partial       *
C      * derivatives, X is a vector of the unknown component           *
C      * concentrations and R is a vector of the residuals.  This eqn  *
C      * will be solved for X.  This X represents the difference       *
C      * between the old guess for the component concentrations and    *
C      * the new guess according to the Newton-Raphson method.  See    *
C      * program documentation for more details.                       *
C      *                                                               *
C      * Variables:                                                    *
C      * A         -Output- A matrix of size "Size" by ("Size"+1).     *
C      *                    The "Size" by "Size" portion of the matrix *
C      *                    contains the coefficents of "Size"         *
C      *                    equations in "Size" unknowns, and the      *
C      *                    "Size"+1st column is the right hand side   *
C      *                    vector of constant terms.  "Size" is equal *
C      *                    to the number of components in the system. *
C      *                    Units are consistent with inputs.          *
C      *                    (Common block VGauss, file VGauss.f).      *
C      * ColA      -Local - Column index into matrix A and vector X.   *
C      *                    Unitless.                                  *
C      * CtoM      -Fcn   - Multiplication factor for conversion from  *
C      *                    concentration to mass.  Units are L.       *
C      * LNum      -Input - The current layer number.                  *
C      * MasBal    -Input - Indicates a species that is limited by     *
C      *                    mass balance.  Unitless.                   *
C      *                    (Common block VModel, file VModel.f)       *
C      * NComp     -Input - The number of components in the system.    *
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * NPhases   -Input - The number of phases in the system.        *
C      *                    Unitless.                                  *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * NSpecies  -Input - The number of species in the system (i.e.  *
C      *                    the number of components plus the number   *
C      *                    of dependent species).  Unitless.          *
C      *                    (Common block VModel, file VModel.f)       *
C      * LIn       -Local - Long array index, i.e. for arrays of       *
C      *                    length NSpecies (components are followed   *
C      *                    by dependent species).  Unitless.          *
C      * Resid     -Input - A vector, of size NComp, of the residual   *
C      *                    for each component.  This residual is the  *
C      *                    difference between the known total         *
C      *                    concentration and the total concentration  *
C      *                    calculated from a set of component         *
C      *                    concentrations.  Units are mol.            *
C      *                    (Common block VModel, file VModel.f)       *
C      * RowA      -Local - Row index into matrix A.  Unitless.        *
C      * S         -Local - Species index.  Unitless.                  *
C      * SC        -Input - A matrix, of size NDepSp by NComp, of the  *
C      *                    stoichiometric coefficient of each         *
C      *                    component for each species.  Unitless.     *
C      *                    (Common block VModel, file VModel.f)       *
C      * SComp     -Input - A vector, of size NPhases, of the          *
C      *                    component associated with each phase.      *
C      *                    Unitless.                                  *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * Size      -Input - The number of chemical components in the   *
C      *                    system.  Unitless.                         *
C      *                    (Common block VGauss, file VGauss.f).      *
C      * SpConc    -Input - A matrix, of size NSpecies by NLayers, of  *
C      *                    species concentrations, one set for each   *
C      *                    layer. Units are mol/L.                    *
C      *                    (Common block VModel, file VModel.f)       *
C      * SPMasBal  -Input - Indicates a solid phase species type       *
C      *                    limited by mass balance.  Unitless.        *
C      *                    (Common block VModel, file VModel.f)       *
C      * SType     -Input - A vector, of size NSpecies, of each        *
C      *                    species's type.  Unitless.                 *
C      *                    (Common block VModel, file VModel.f)       *
C      * Sum       -Local - Summation term in calculation.  Units      *
C      *                    consistent with equation.                  *
C      * tSC       -Input - A matrix, of size NSpecies by NComp, of    *
C      *                    the transformed stoichiometric coefficients*
C      *                    of each comp for each species.  Unitless.  *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      *****************************************************************
	      SUBROUTINE JAKE(LNUM)
							IMPLICIT NONE
							INCLUDE 'VGAUSS.INC'
							INCLUDE 'VMODEL.INC'
       REAL*8 CTOM, T1, T2, SUM
       INTEGER B, J, K, LNUM, S, P, Q

C      *------------------------------------------------------*
C      * Set the size of the matrix for the gaussian routine. *	   
C      *------------------------------------------------------*
       SIZE = NCOMP

C      *-------------------------------*	   
C      * Create the matrix row by row. *
C      *-------------------------------*	   
       DO 400 J = 1, NCOMP

C         *----------------------------------------------------------*
C         * For each column, that is, for each component, calculate  *
C         * the change in the total concentration of the row         *
C         * component caused by a change in the concentration of     *
C         * the column component.                                    *
C         * The calculation is done in four parts.  First the        *
C         * summation term is calculated.  Second, the summation is  *
C         * divided by the constant.  Third, the phase contribution  *
C         * is added in.  Fourth, the diagonal terms, where the row  *
C         * and column components are the same are adjusted by       *
C         * adding CtoM to them.  For details of the derivation see  *
C         * program documentation.                                   *
C         *----------------------------------------------------------*
          DO 300 K = 1, NCOMP
             IF ((STYPE(J).EQ.MASBAL).OR.(STYPE(J).EQ.SPMASBAL)) THEN

C               *---------------------------------*
C               * Calculate species contribution. *
C               *---------------------------------*
                SUM = 0.0
                DO 100 S = NCOMP + 1, NSPECIES
                     SUM = SUM + TSC(S, K) * TSC(S, J) * 
     >                    SPCONC(S, LNUM) * CTOM(S,LNUM)
 100            CONTINUE

C               *--------------------------------------------*
C               * Divide out component concentration toward  *
C               * calculating derivative.                    *
C               *--------------------------------------------*      
                IF (SPCONC(K, LNUM).NE.0) THEN
                   A(J, K) = SUM / SPCONC(K,LNUM)
                END IF

C               *---------------------------------*
C               * Calculate phases contributions. *
C               *---------------------------------*
                SUM = 0.0
                DO 200 Q =1, NPHASES
                   P = SORDER(Q)
                   B = SCOMP(P)
                   T1 = SPCONC(B,LNUM) * CTOM(B,LNUM) / 
     >                  SPCONC(K, LNUM)
                   T2 = TSC(B, K) * TSC(B, J) 
                   SUM = SUM + T1 * T2
  200           CONTINUE
                A(J, K) = A(J, K) + SUM
             ELSE
                A(J,K) = 0.0
             ENDIF
  300     CONTINUE
C         *--------------------------*
C         * Adjust the diagonal term *
C         *--------------------------*
          A(J, J) = A(J, J) + CTOM(J,LNUM)

C         *---------------------------------------------------------*
C         * Augment the Jacobian with the right hand side vector of *
C         * constant terms, that is, the residuals.                 *
C         *---------------------------------------------------------*
          A(J, NCOMP+1) = RESID(J)
  400  CONTINUE

       RETURN
       END
C      ****************************************************************
C      *               END OF SUBROUTINE JACOBIAN
C      ****************************************************************
